<?php
// handy_script_handler.php - Optimized version with cleanup

// Set appropriate headers for the API response
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Configuration
$tempDir = 'temp/';   // Temporary storage for original files
$csvDir = 'csv/';     // CSV files for Handy
$baseUrl = 'https://vector.slx.my/';
$debugLog = true;
$fileRetentionHours = 1; // How long to keep files on server

// Create directories if they don't exist
if (!file_exists($tempDir)) {
    mkdir($tempDir, 0755, true);
}
if (!file_exists($csvDir)) {
    mkdir($csvDir, 0755, true);
}

// Log function
function logMessage($message) {
    global $debugLog;
    if ($debugLog) {
        file_put_contents('handy_debug.log', 
            date('Y-m-d H:i:s') . " - $message\n", 
            FILE_APPEND);
    }
}

// Run cleanup of old files
cleanupOldFiles();

// Handle direct CSV file access
if (isset($_GET['file']) && !empty($_GET['file'])) {
    $filename = basename($_GET['file']);
    $filepath = $csvDir . $filename;
    
    if (file_exists($filepath) && pathinfo($filepath, PATHINFO_EXTENSION) === 'csv') {
        // Set proper headers for CSV serving
        header('Content-Type: text/csv');
        header('Access-Control-Allow-Origin: *');
        header('Content-Length: ' . filesize($filepath));
        
        // Log access
        logMessage("Serving CSV file: $filename, size: " . filesize($filepath) . " bytes");
        
        // Output the file content
        readfile($filepath);
        exit;
    } else {
        header('HTTP/1.0 404 Not Found');
        echo 'CSV file not found';
        logMessage("ERROR: CSV file not found: $filename");
        exit;
    }
}

// Handle OPTIONS preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type');
    header('Access-Control-Max-Age: 86400');
    exit(0);
}

// Process file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['script'])) {
    $uploadedFile = $_FILES['script'];
    logMessage("Received upload: {$uploadedFile['name']}, size: {$uploadedFile['size']} bytes");
    
    // Error handling for file upload
    if ($uploadedFile['error'] !== UPLOAD_ERR_OK) {
        $error = getUploadErrorMessage($uploadedFile['error']);
        logMessage("Upload error: $error");
        sendResponse(false, "Upload error: $error");
        exit;
    }
    
    // Get the file extension
    $fileExt = strtolower(pathinfo($uploadedFile['name'], PATHINFO_EXTENSION));
    logMessage("File extension: $fileExt");
    
    // Generate a unique filename for the uploaded file
    $uniqueFilename = uniqid('temp_') . '.' . $fileExt;
    $uploadPath = $tempDir . $uniqueFilename;
    
    // Move the uploaded file to our temporary directory
    if (!move_uploaded_file($uploadedFile['tmp_name'], $uploadPath)) {
        logMessage("Failed to save uploaded file");
        sendResponse(false, 'Failed to save the uploaded file');
        exit;
    }
    
    logMessage("File temporarily saved to: $uploadPath");
    
    // Process the file based on extension
    if ($fileExt === 'funscript' || $fileExt === 'json') {
        try {
            // Read the file content
            $jsonContent = file_get_contents($uploadPath);
            logMessage("Read JSON content, length: " . strlen($jsonContent));
            
            $funscript = json_decode($jsonContent, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                // Clean up the temporary file on error
                if (file_exists($uploadPath)) {
                    unlink($uploadPath);
                    logMessage("Deleted temporary file due to error: $uploadPath");
                }
                
                logMessage("Invalid JSON format: " . json_last_error_msg());
                sendResponse(false, 'Invalid JSON format in funscript');
                exit;
            }
            
            if (!isset($funscript['actions']) || !is_array($funscript['actions'])) {
                // Clean up the temporary file on error
                if (file_exists($uploadPath)) {
                    unlink($uploadPath);
                    logMessage("Deleted temporary file due to error: $uploadPath");
                }
                
                logMessage("Missing actions array in funscript");
                sendResponse(false, 'Invalid funscript format - missing actions array');
                exit;
            }
            
            $actionCount = count($funscript['actions']);
            logMessage("Funscript contains $actionCount actions");
            
            // Generate a unique CSV filename
            $csvFilename = uniqid('script_') . '.csv';
            $csvPath = $csvDir . $csvFilename;
            
            // Convert funscript to CSV
            $csvContent = exactFunscriptIOConversion($funscript);
            logMessage("Generated CSV content, length: " . strlen($csvContent));
            
            // Log the first few lines of CSV for debugging
            $csvLines = explode("\n", $csvContent);
            $csvPreview = implode("\n", array_slice($csvLines, 0, min(5, count($csvLines))));
            logMessage("CSV preview:\n$csvPreview");
            
            // Save the CSV file
            $writeResult = file_put_contents($csvPath, $csvContent);
            if ($writeResult === false) {
                // Clean up the temporary file on error
                if (file_exists($uploadPath)) {
                    unlink($uploadPath);
                    logMessage("Deleted temporary file due to error: $uploadPath");
                }
                
                logMessage("Failed to save CSV file");
                sendResponse(false, 'Failed to save CSV file');
                exit;
            }
            
            logMessage("CSV saved to: $csvPath, size: " . filesize($csvPath));
            
            // Delete the temporary original file now that we have the CSV
            if (file_exists($uploadPath)) {
                unlink($uploadPath);
                logMessage("Deleted temporary file after successful conversion: $uploadPath");
            }
            
            // Return the direct URL to the CSV file
            $csvUrl = $baseUrl . $csvDir . $csvFilename;
            logMessage("Success! Returning URL: $csvUrl");
            
            sendResponse(true, 'Script converted successfully', $csvUrl);
            
        } catch (Exception $e) {
            // Clean up the temporary file on error
            if (file_exists($uploadPath)) {
                unlink($uploadPath);
                logMessage("Deleted temporary file due to exception: $uploadPath");
            }
            
            logMessage("Exception: " . $e->getMessage());
            sendResponse(false, 'Error processing funscript: ' . $e->getMessage());
        }
    } else {
        // If file is already a CSV, just copy it to the CSV directory
        if ($fileExt === 'csv') {
            $csvFilename = uniqid('script_') . '.csv';
            $csvPath = $csvDir . $csvFilename;
            
            if (copy($uploadPath, $csvPath)) {
                // Delete the temporary original file after copy
                if (file_exists($uploadPath)) {
                    unlink($uploadPath);
                    logMessage("Deleted temporary file after successful copy: $uploadPath");
                }
                
                $csvUrl = $baseUrl . $csvDir . $csvFilename;
                logMessage("Copied existing CSV file to: $csvPath");
                sendResponse(true, 'CSV file uploaded successfully', $csvUrl);
            } else {
                // Clean up the temporary file on error
                if (file_exists($uploadPath)) {
                    unlink($uploadPath);
                    logMessage("Deleted temporary file due to copy error: $uploadPath");
                }
                
                logMessage("Failed to copy CSV file");
                sendResponse(false, 'Failed to copy CSV file');
            }
        } else {
            // Clean up the temporary file for unsupported formats
            if (file_exists($uploadPath)) {
                unlink($uploadPath);
                logMessage("Deleted temporary file with unsupported format: $uploadPath");
            }
            
            logMessage("Unsupported file format: $fileExt");
            sendResponse(false, 'Unsupported file format. Please upload a funscript, JSON, or CSV file.');
        }
    }
} else {
    // If it's not a proper request, show info or error
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo "<html><body><h1>Handy Script Handler</h1>";
        echo "<p>This endpoint accepts funscript uploads via POST and serves CSV files to the Handy device.</p>";
        echo "</body></html>";
    } else {
        logMessage("Invalid request: No file uploaded");
        sendResponse(false, 'No file uploaded or invalid request method');
    }
}

/**
 * Clean up files older than the retention period
 */
function cleanupOldFiles() {
    global $tempDir, $csvDir, $fileRetentionHours;
    
    // Current time
    $now = time();
    $retentionPeriod = $fileRetentionHours * 3600; // Convert hours to seconds
    
    // Clean up temporary directory
    if (file_exists($tempDir) && is_dir($tempDir)) {
        $files = glob($tempDir . '*');
        foreach ($files as $file) {
            $lastModified = filemtime($file);
            if ($now - $lastModified > $retentionPeriod) {
                unlink($file);
                logMessage("Cleanup: Deleted old temporary file: " . basename($file));
            }
        }
    }
    
    // Clean up CSV directory
    if (file_exists($csvDir) && is_dir($csvDir)) {
        $files = glob($csvDir . '*');
        foreach ($files as $file) {
            $lastModified = filemtime($file);
            if ($now - $lastModified > $retentionPeriod) {
                unlink($file);
                logMessage("Cleanup: Deleted old CSV file: " . basename($file));
            }
        }
    }
}

/**
 * Exact conversion function matching funscript-io-2 implementation
 */
function exactFunscriptIOConversion($funscript) {
    // EXACTLY match the funscript-io-2 header format
    $csv = '"{""type"":""handy""}",'.PHP_EOL;
    
    // Must sort actions by time as in funscript-io-2
    $actions = $funscript['actions'];
    usort($actions, function($a, $b) {
        return $a['at'] - $b['at'];
    });
    
    // Check if this script uses inverted positions
    $isInverted = isset($funscript['inverted']) && $funscript['inverted'] === true;
    
    // Empty script check
    if (count($actions) === 0) {
        // Empty script, add a safe default
        $csv .= "0,50".PHP_EOL;  // Changed from 500 to 50
        logMessage("Empty script, added safe default position");
        return $csv;
    }
    
    // Get the first action details
    $firstAction = $actions[0];
    $firstTime = $firstAction['at'];
    $firstPos = $firstAction['pos'];
    
    // Apply inversion if needed
    if ($isInverted) {
        $firstPos = 0 - $firstPos;
    }
    
    // CRITICAL CHANGE: Do NOT multiply by 10 - use positions directly
    $firstHandyPos = $firstPos;
    
    // Ensure within bounds (0-100) instead of (0-1000)
    $firstHandyPos = max(0, min(100, $firstHandyPos));
    
    // CRITICAL: Add a position at time 0 with same position as first action
    $csv .= "0,$firstHandyPos".PHP_EOL;
    logMessage("Added initial position at time 0: $firstHandyPos (same as first action position)");
    
    // Process each action
    foreach ($actions as $action) {
        $time = $action['at'];
        $pos = $action['pos'];
        
        // Apply inversion if needed
        if ($isInverted) {
            $pos = 100 - $pos;
        }
        
        // CRITICAL CHANGE: Do NOT multiply by 10 - use positions directly
        $handyPos = $pos;
        
        // Ensure within bounds (0-100) instead of (0-1000)
        $handyPos = max(0, min(100, $handyPos));
        
        // Add to CSV
        $csv .= "$time,$handyPos".PHP_EOL;
    }
    
    return $csv;
}

/**
 * Send a JSON response back to the client
 */
function sendResponse($success, $message, $url = null) {
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($url !== null) {
        $response['url'] = $url;
    }
    
    echo json_encode($response);
}

/**
 * Get error message for file upload errors
 */
function getUploadErrorMessage($errorCode) {
    switch ($errorCode) {
        case UPLOAD_ERR_INI_SIZE:
            return 'The uploaded file exceeds the upload_max_filesize directive in php.ini';
        case UPLOAD_ERR_FORM_SIZE:
            return 'The uploaded file exceeds the MAX_FILE_SIZE directive in the HTML form';
        case UPLOAD_ERR_PARTIAL:
            return 'The uploaded file was only partially uploaded';
        case UPLOAD_ERR_NO_FILE:
            return 'No file was uploaded';
        case UPLOAD_ERR_NO_TMP_DIR:
            return 'Missing a temporary folder';
        case UPLOAD_ERR_CANT_WRITE:
            return 'Failed to write file to disk';
        case UPLOAD_ERR_EXTENSION:
            return 'A PHP extension stopped the file upload';
        default:
            return 'Unknown upload error';
    }
}